/*=========================================================================

  Program:   ORFEO Toolbox
  Language:  C++
  Date:      $Date$
  Version:   $Revision$


  Copyright (c) Centre National d'Etudes Spatiales. All rights reserved.
  See OTBCopyright.txt for details.


     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notices for more information.

=========================================================================*/
#if defined(_MSC_VER)
#pragma warning ( disable : 4786 )
#endif

#include "itkFixedArray.h"
#include "otbImage.h"
#include "otbImageFileReader.h"
#include "otbImageFileWriter.h"
#include "otbGPUFineCorrelationImageFilter.h"
#include "otbStandardFilterWatcher.h"
#include "otbExtractROI.h"
#include "itkTranslationTransform.h"
#include "otbStreamingResampleImageFilter.h"

// ./FineCorrelationGPU StereoFixed.png StereoMoving.png outputcorrel.tif outputfield.tif 5 10 0 0.01
// ./FineCorrelationGPU StereoFixed.png StereoMoving.png outputcorrel.tif outputfield.tif 5 10 1 0.01
// ./FineCorrelationGPU StereoFixed.png StereoMoving.png outputcorrel.tif outputfield.tif 5 10 2 0.01 => does not work


// ./FineCorrelationGPU ~/OTB/trunk/OTB-Data/LargeInput/QUICKBIRD/TOULOUSE/000000128955_01_P001_PAN/02APR01105228-P1BS-000000128955_01_P001.TIF 100 100 outputcorrelGPU.tif outputfieldGPU.tif 5 15 1 100

int main( int argc, char * argv[] )
{
  if(argc < 9)
    {
    std::cerr<<"Usage: "<< argv[0] <<" fixed_fname moving_fname output_correl output_field"
        << " radius search_radius refineMethod subpixPrecision \n"
        << " or (for simulations)\n"
        << argv[0] <<" fixed_fname extract_sizeX extract_sizeY output_correl output_field"
        << " radius search_radius refineMethod subpixPrecision \n";

    return EXIT_FAILURE;
    }
  const char * fixedFileName;
  const char * movingFileName;
  const char * correlFileName;
  const char * fieldFileName;
  unsigned int radius;
  unsigned int sradius;
  int          refine;
  unsigned int precision;
  unsigned int extractSizeX;
  unsigned int extractSizeY;
  bool simulatedMoving = false;

  if (argc == 9)
    {
    fixedFileName = argv[1];
    movingFileName = argv[2];
    correlFileName = argv[3];
    fieldFileName = argv[4];
    radius = atoi(argv[5]);
    sradius = atoi(argv[6]);
    refine = atoi(argv[7]);
    precision = atoi(argv[8]);
    }
  else
    {
    fixedFileName = argv[1];
    extractSizeX= atoi(argv[2]);
    extractSizeY= atoi(argv[3]);
    correlFileName = argv[4];
    fieldFileName = argv[5];
    radius = atoi(argv[6]);
    sradius = atoi(argv[7]);
    refine = atoi(argv[8]);
    precision = atoi(argv[9]);
    simulatedMoving = true;
    }

  typedef float      PixelType;
  const unsigned int  Dimension = 2;

  typedef itk::FixedArray<PixelType,Dimension>                                DeformationValueType;
  typedef otb::Image< PixelType,  Dimension >                                 ImageType;
  typedef otb::Image<DeformationValueType,Dimension>                          FieldImageType;
  typedef otb::ImageFileReader< ImageType >                                   ReaderType;
  typedef otb::ImageFileWriter< ImageType >                                   CorrelWriterType;
  typedef otb::ImageFileWriter< FieldImageType>                               FieldWriterType;
  typedef otb::GPUFineCorrelationImageFilter<ImageType,ImageType,FieldImageType> CorrelationFilterType;

  typedef otb::ExtractROI<PixelType, PixelType> ExtractROIFilterType;


  ReaderType::Pointer freader = ReaderType::New();
  freader->SetFileName(fixedFileName);
  
  CorrelationFilterType::Pointer correlation = CorrelationFilterType::New();

  ReaderType::Pointer mreader = ReaderType::New();

  ExtractROIFilterType::Pointer fixROIFilter = ExtractROIFilterType::New();
  ExtractROIFilterType::Pointer movROIFilter = ExtractROIFilterType::New();


  if (simulatedMoving)
    {
    mreader->SetFileName(fixedFileName);//Note necessary to have two different reader to maintain on demand processing

    fixROIFilter->SetStartX(2000);
    fixROIFilter->SetStartY(2000);
    fixROIFilter->SetSizeX(extractSizeX);
    fixROIFilter->SetSizeY(extractSizeY);
    fixROIFilter->SetInput(freader->GetOutput());
    fixROIFilter->Update();//FIXME

    movROIFilter->SetStartX(1995);
    movROIFilter->SetStartY(2008);
    movROIFilter->SetSizeX(extractSizeX);
    movROIFilter->SetSizeY(extractSizeY);
    movROIFilter->SetInput(mreader->GetOutput());
    movROIFilter->Update();//FIXME

    correlation->SetFixedInput(fixROIFilter->GetOutput());
    correlation->SetMovingInput(movROIFilter->GetOutput());
    }
  else
    {
    mreader->SetFileName(movingFileName);
    correlation->SetFixedInput(freader->GetOutput());
    correlation->SetMovingInput(mreader->GetOutput());
    }



  correlation->SetRadius(radius);
  correlation->SetSearchRadius(sradius);
  if(refine == 0)
    {
    correlation->SetRefinementModeToCoarse();
    }
  else if(refine == 1)
    {
    correlation->SetRefinementModeToLSQRQuadFit();
    }
  else
    {
    correlation->SetRefinementModeToSubPixel();
    }

  correlation->SetSubPixelPrecision(precision);
//  correlation->SetNumberOfThreads(1);

  otb::StandardFilterWatcher watcher(correlation,"Correlation");

  timespec startClock, endClock;
  time_t startTime = time(NULL);
  clock_t startDensity = clock();
  clock_gettime(CLOCK_REALTIME, &startClock);

  correlation->Update();

  clock_gettime(CLOCK_REALTIME, &endClock);
  clock_t endDensity = clock();
  time_t endTime = time(NULL);

  std::cout << "Time1: " << std::setprecision(15)
            << (endDensity-startDensity)/((float) CLOCKS_PER_SEC) << std::endl;
  std::cout << "Time2: " << std::setprecision(15)
            << (endTime-startTime) << std::endl;
  std::cout << "Time3: " << std::setprecision(15)
            << (endClock.tv_sec-startClock.tv_sec) + (endClock.tv_nsec-startClock.tv_nsec)/1000000000. << std::endl;



  CorrelWriterType::Pointer correlWriter = CorrelWriterType::New();
  correlWriter->SetFileName(correlFileName);
  correlWriter->SetInput(correlation->GetOutput());
  correlWriter->Update();

  FieldWriterType::Pointer fieldWriter = FieldWriterType::New();
  fieldWriter->SetFileName(fieldFileName);
  fieldWriter->SetInput(correlation->GetOutputDeformationField());
  fieldWriter->Update();

  return EXIT_SUCCESS;
}
