
#include <cstdlib>

#include "itkPointSet.h"
#include "otbRandomPointSetSource.h"
#include "otbImage.h"
#include "otbGPUPointSetToDensityImageFilter.h"
#include "otbStreamingStatisticsImageFilter.h"
#include "otbImageFileWriter.h"
#include "itkScalarToRGBColormapImageFilter.h"
#include "otbScalarToRainbowRGBPixelFunctor.h"
#include "otbPointSetDensityGaussianFunction.h"
#include "otbPointSetDensityEpanechnikovFunction.h"

// Command line for 1000 points in an image of 1000x1000
// ./densityGPU 1000 1000 1000 output-gpu.png

int main(int argc, char * argv[])
{
  if (argc != 5)
    {
    std::cerr << "Usage: " << argv[0]
              << " <#points> <size X> <size Y> <output>" << std::endl;
    return EXIT_FAILURE;
    }

  unsigned int nPoints = atoi(argv[1]);
  int sizeX = atoi(argv[2]);
  int sizeY = atoi(argv[3]);
  std::string filename = argv[4];

  std::cout << "Compute with:\n"
      << " * " << nPoints << " points\n"
      << " * " << sizeX << ", " << sizeY << " image"
      << " * " << "output in " << filename << std::endl;

  typedef float                                   PixelType;
  typedef itk::PointSet<PixelType, 2>             PointSetType;
  typedef otb::RandomPointSetSource<PointSetType> PointSetSource;
  typedef otb::Image<PixelType, 2>                ImageType;

  PointSetSource::Pointer pointSetSource = PointSetSource::New();
  pointSetSource->SetNumberOfPoints(nPoints);
  PointSetType::PointType maxPoint;
  maxPoint[0] = sizeX;
  maxPoint[1] = sizeY;
  pointSetSource->SetMaxPoint(maxPoint);
  pointSetSource->SetSeed(152002);

  pointSetSource->Update();


  typedef otb::GPUPointSetToDensityImageFilter<PointSetType, ImageType> PSDensityFilterType;
  PSDensityFilterType::Pointer density = PSDensityFilterType::New();
  density->SetInput(pointSetSource->GetOutput());
  density->SetRadius(25);
  ImageType::SizeType size;
  size[0] = sizeX;
  size[1] = sizeY;
  density->SetSize(size);

  timespec startClock, endClock;
  time_t startTime = time(NULL);
  clock_t startDensity = clock();
  clock_gettime(CLOCK_REALTIME, &startClock);

  density->Update();

  clock_gettime(CLOCK_REALTIME, &endClock);
  clock_t endDensity = clock();
  time_t endTime = time(NULL);

  std::cout << "Time1: " << std::setprecision(15)
            << (endDensity-startDensity)/((float) CLOCKS_PER_SEC) << std::endl;
  std::cout << "Time2: " << std::setprecision(15)
            << (endTime-startTime) << std::endl;
  std::cout << "Time3: " << std::setprecision(15)
            << (endClock.tv_sec-startClock.tv_sec) + (endClock.tv_nsec-startClock.tv_nsec)/1000000000. << std::endl;

  typedef otb::StreamingStatisticsImageFilter<ImageType> StatFilterType;
  StatFilterType::Pointer stat = StatFilterType::New();
  stat->SetInput(density->GetOutput());
  stat->Update();
  double min = stat->GetMinimum();
  double max = stat->GetMaximum();

  typedef unsigned char                               UCharPixelType;
  typedef itk::RGBPixel<UCharPixelType>               RGBPixelType;
  typedef otb::Image<RGBPixelType, 2>                 RGBImageType;
  typedef itk::ScalarToRGBColormapImageFilter<ImageType,RGBImageType> ColorMapFilterType;
  ColorMapFilterType::Pointer colormapper = ColorMapFilterType::New();
  typedef otb::Functor::ScalarToRainbowRGBPixelFunctor<PixelType, RGBPixelType> ColorMapFunctorType;
  ColorMapFunctorType::Pointer colormap = ColorMapFunctorType::New();
  colormapper->SetColormap(colormap);
  colormap->SetMinimumInputValue(min);
  colormap->SetMaximumInputValue(max);
  colormapper->UseInputImageExtremaForScalingOff();
  colormapper->SetInput(density->GetOutput());

  typedef otb::ImageFileWriter<RGBImageType> WriterType;
  WriterType::Pointer writer = WriterType::New();
  writer->SetFileName(filename);
  writer->SetInput(colormapper->GetOutput());
  writer->Update();

  return EXIT_SUCCESS;
}
