/*=========================================================================

  Program:   ORFEO Toolbox
  Language:  C++
  Date:      $Date$
  Version:   $Revision$


  Copyright (c) Centre National d'Etudes Spatiales. All rights reserved.
  See OTBCopyright.txt for details.


     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notices for more information.

=========================================================================*/
#ifndef __otbGPUFineCorrelationImageFilter_h
#define __otbGPUFineCorrelationImageFilter_h

#include "otbFineCorrelationImageFilter.h"

namespace otb
{

/** \class GPUFineCorrelationImageFilter
 * \brief Computes a deformation field between two images
 *
 * This class compute a correlation field and the associated
 * deformation field between two images.
 *
 * The m_Radius parameter defines the size of the window to compute
 * the normalized correlation.
 *
 * The m_SearchRadius parameter defines the size of the area where to
 * search for a correlation peak in the moving image.
 *
 * Once the correlation peak has been found, it can be further refined
 * by trying to fit a quadric surface in the premises of the
 * maxima, thus obtaining sub-pixel precision. This option is
 * activated by setting m_RefineLocation to true.
 *
 * TOutputCorrelation is supposed to be a scalar image with floating
 * point precision.
 *
 * TOutputDeformationField is supposed to be an image whose pixel is a
 * fixed size vector of size 2, with floating point precision.
 *
 * Images spacing is ignored during computation and in the output
 * deformation field.
 *
 * All computation are done in double precision.
 * 
 * \ingroup IntensityImageFilters   Multithreaded, Streamed
 *
 */
template <class TInputImage, class T0utputCorrelation, class TOutputDeformationField>
class ITK_EXPORT GPUFineCorrelationImageFilter : public FineCorrelationImageFilter<TInputImage,T0utputCorrelation,TOutputDeformationField>
{
public:
  /** Standard class typedefs. */
  typedef GPUFineCorrelationImageFilter                          Self;
  typedef FineCorrelationImageFilter<TInputImage,
                    T0utputCorrelation,TOutputDeformationField>  Superclass;
  typedef itk::SmartPointer<Self>                                Pointer;
  typedef itk::SmartPointer<const Self>                          ConstPointer;

  /** Method for creation through the object factory. */
  itkNewMacro(Self);

  /** Run-time type information (and related methods). */
  itkTypeMacro(GPUFineCorrelationImageFilter, FineCorrelationImageFilter);

  /** Some convenient typedefs. */
  typedef TInputImage InputImageType;
  typedef typename T0utputCorrelation::RegionType                 OutputImageRegionType;
  typedef typename TOutputDeformationField::PixelType             DeformationValueType;
  typedef typename TInputImage::Pointer                           InputImagePointerType;
  typedef typename TInputImage::RegionType                        InputImageRegionType;
  typedef typename TInputImage::SizeType                          SizeType;
  typedef typename TInputImage::IndexType                         IndexType;
  typedef typename itk::ImageRegionIterator<TInputImage>          InputImageRegionIteratorType;
  typedef typename itk::InterpolateImageFunction
  <TInputImage, double>                                           InterpolatorType;
  typedef typename InterpolatorType::Pointer                      InterpolatorPointerType;
  typedef typename itk::ContinuousIndex<double, 2>                ContinuousIndexType;

  /** Typedef for refinements mode */
  typedef enum {COARSE = 0, LSQR_QUADFIT = 1, SUBPIXEL = 2}       RefinementModeType; //FIXME to factorize with the superclass

 
  /** Typedef of the input iterator */
  typedef itk::ConstNeighborhoodIterator<TInputImage>             NeighborhoodIteratorType;  
  typedef typename NeighborhoodIteratorType::RadiusType           RadiusType;
  typedef typename NeighborhoodIteratorType::NeighborhoodType     NeighborhoodType;
  typedef typename NeighborhoodIteratorType::OffsetType           OffsetType;
  
  typedef itk::ImageRegionConstIterator<InputImageType>           ImageConstIterator;
  typedef itk::ImageRegionIterator<InputImageType>                ImageIterator;

protected:
  /** Constructor */
  GPUFineCorrelationImageFilter();
  /** Destructor */
  virtual ~GPUFineCorrelationImageFilter() {};

  /** Threaded generate data */
  virtual void ThreadedGenerateData(const OutputImageRegionType& outputRegionForThread,int threadId );


private:
  GPUFineCorrelationImageFilter(const Self&); //purposely not implemented
  void operator=(const Self&); //purposely not implemented


};

} // end namespace otb

#ifndef OTB_MANUAL_INSTANTIATION
#include "otbGPUFineCorrelationImageFilter.txx"
#endif

#endif
