#!/bin/sh
#
# Copyright (C) 2005-2024 Centre National d'Etudes Spatiales (CNES)
#
# This file is part of Orfeo Toolbox
#
#     https://www.orfeo-toolbox.org/
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

# avoid magic number. Version filled by cmake
OTB_SHORT_VERSION=9.1

echo "***** First time launching OTB after installation, doing some post installation steps before use *****"
# Apply necessary patches for a modular install because cmake generates these file at configure time, not at packaging time
CMAKE_DIRS="$OTB_INSTALL_DIR/lib/cmake"
sed -i "s/FATAL_ERROR/WARNING/g" "$CMAKE_DIRS/OTB-$OTB_SHORT_VERSION/OTBTargets.cmake"
sed -i "s/FATAL_ERROR/WARNING/g" "$CMAKE_DIRS/OTB-$OTB_SHORT_VERSION/OTBModuleAPI.cmake"
sed -i "s/\/builds\/otb\/xdk/\${OTB_INSTALL_PREFIX}/g" "$CMAKE_DIRS"/OTB-"$OTB_SHORT_VERSION"/*.cmake
sed -i "s/\/builds\/otb\/xdk/\${OTB_INSTALL_PREFIX}/g" "$CMAKE_DIRS"/OTB-"$OTB_SHORT_VERSION"/Modules/*.cmake
sed -i "s/\/builds\/otb\/xdk/\${OTB_INSTALL_PREFIX}/g" "$CMAKE_DIRS"/ITK-4.13/*.cmake
sed -i "s/\/builds\/otb\/xdk/\${OTB_INSTALL_PREFIX}/g" "$CMAKE_DIRS"/ITK-4.13/Modules/*.cmake
sed -i "s/\/builds\/otb\/xdk/\$OTB_INSTALL_DIR/g" "$OTB_INSTALL_DIR"/bin/gdal-config
sed -i "s/\/builds\/otb\/xdk/\$OTB_INSTALL_DIR/g" "$OTB_INSTALL_DIR"/bin/curl-config
sh "$OTB_INSTALL_DIR"/tools/sanitize_rpath.sh

ostype="$(lsb_release -is)"

if [ "$ostype" != "RedHatEnterprise" ] ; then
  concatpyversion="$(python3 -V 2>&1 | sed 's/.* \([0-9]\).\([0-9]*\).*/\1\2/')"
  # Compare the concat version with the version used to build this OTB package
  # The python build version is filled by cmake
  if [ "$concatpyversion" -ne "38" ]; then
    sh "$OTB_INSTALL_DIR"/recompile_bindings.sh
  fi
  # Check python version, if python 3.12 (ubuntu 24 and debian > 12) download
  # and extract the gdal bindings for python 3.12
  # In case your install is from compiled code, the gdal bindings will be
  # already there in the installation
  # Note that the install path differs from OS. To handle it,
  # check before the folder name (always prefixed of python3)
  osgeo_pkg_path=$(find "$OTB_INSTALL_DIR/lib" -mindepth 3 -maxdepth 3 -type d -name "osgeo")
  gdal_python_found="$(find "$osgeo_pkg_path" -name "*$concatpyversion-x86_64-linux*")"
  if [ -z "$gdal_python_found" ]; then
    echo "***** Python $concatpyversion detected, downloading gdal bindings compiled for this Python version *****"
    HTTP_STATUS=$(curl -s -o "$OTB_INSTALL_DIR"/tools/gdal-py"$concatpyversion".tar.gz -w "%{response_code}\n" https://www.orfeo-toolbox.org/packages/archives/OTB/OTB-"$OTB_SHORT_VERSION"-GDAL-bindings-py"$concatpyversion".tar.gz)
    if [ "$HTTP_STATUS" -eq 200 ]; then
      tar -xf "$OTB_INSTALL_DIR"/tools/gdal-py"$concatpyversion".tar.gz -C "$osgeo_pkg_path"
      rm "$OTB_INSTALL_DIR"/tools/gdal-py"$concatpyversion".tar.gz
      echo "***** GDAL bindings for python $concatpyversion successfully installed *****"
    else
      echo "Can not find GDAL bindings at https://www.orfeo-toolbox.org/packages/archives/OTB/OTB-$OTB_SHORT_VERSION-GDAL-bindings-py$concatpyversion.tar.gz"
      return 255
    fi
  fi
fi

echo "***** OTB Environment and Python bindings setup complete *****"
echo "OK" > "$OTB_INSTALL_DIR"/tools/install_done.txt
